<?php
// Get notices
function get_active_notices() {
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM notices WHERE is_active = 1 ORDER BY created_at DESC");
    $stmt->execute();
    return $stmt->get_result();
}

// Add new notice
function add_notice($title, $message) {
    global $conn;
    $stmt = $conn->prepare("INSERT INTO notices (title, message) VALUES (?, ?)");
    $stmt->bind_param("ss", $title, $message);
    return $stmt->execute();
}

// Update notice
function update_notice($id, $title, $message, $is_active) {
    global $conn;
    $stmt = $conn->prepare("UPDATE notices SET title = ?, message = ?, is_active = ? WHERE id = ?");
    $stmt->bind_param("ssii", $title, $message, $is_active, $id);
    return $stmt->execute();
}

// Delete notice
function delete_notice($id) {
    global $conn;
    $stmt = $conn->prepare("DELETE FROM notices WHERE id = ?");
    $stmt->bind_param("i", $id);
    return $stmt->execute();
}

// Get all notices (for admin)
function get_all_notices() {
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM notices ORDER BY created_at DESC");
    $stmt->execute();
    return $stmt->get_result();
}

// Authentication functions
function login_user($email, $password) {
    global $conn;

    $stmt = $conn->prepare("SELECT id, password FROM users WHERE email = ?");
    $stmt->bind_param("s", $email);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows == 1) {
        $user = $result->fetch_assoc();
        if (password_verify($password, $user['password'])) {
            $_SESSION['user_id'] = $user['id'];
            return true;
        }
    }
    return false;
}

function register_user($full_name, $email, $phone, $password, $referrer_code = null) {
    global $conn;

    // Check if email or phone already exists
    $stmt = $conn->prepare("SELECT id FROM users WHERE email = ? OR phone = ?");
    $stmt->bind_param("ss", $email, $phone);
    $stmt->execute();
    if ($stmt->get_result()->num_rows > 0) {
        return ['success' => false, 'message' => 'ইমেইল বা ফোন নম্বর ইতিমধ্যে নিবন্ধিত।'];
    }

    // Find referrer
    $referrer_id = null;
    if ($referrer_code) {
        $stmt = $conn->prepare("SELECT id FROM users WHERE referral_code = ?");
        $stmt->bind_param("s", $referrer_code);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $referrer_id = $result->fetch_assoc()['id'];
        }
    }

    // Generate referral code
    $referral_code = generate_referral_code();

    // Hash password
    $hashed_password = password_hash($password, PASSWORD_DEFAULT);

    // Insert user
    $stmt = $conn->prepare("INSERT INTO users (full_name, email, phone, password, referral_code, referrer_id) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("sssssi", $full_name, $email, $phone, $hashed_password, $referral_code, $referrer_id);

    if ($stmt->execute()) {
        $user_id = $conn->insert_id;
        $_SESSION['user_id'] = $user_id;
        return ['success' => true, 'message' => 'রেজিস্ট্রেশন সফল হয়েছে।'];
    }

    return ['success' => false, 'message' => 'রেজিস্ট্রেশন ব্যর্থ হয়েছে।'];
}

function logout_user() {
    session_destroy();
    header("Location: index.php");
    exit();
}

// Investment functions
function get_companies_by_category($category) {
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM companies WHERE category = ? ORDER BY name");
    $stmt->bind_param("s", $category);
    $stmt->execute();
    return $stmt->get_result();
}

function calculate_daily_profit($amount, $return_percentage) {
    return ($amount * $return_percentage) / 100;
}

function create_investment($user_id, $company_id, $amount) {
    global $conn;

    // Get company details
    $stmt = $conn->prepare("SELECT return_percentage FROM companies WHERE id = ?");
    $stmt->bind_param("i", $company_id);
    $stmt->execute();
    $company = $stmt->get_result()->fetch_assoc();

    $profit_per_day = calculate_daily_profit($amount, $company['return_percentage']);

    // Insert investment
    $stmt = $conn->prepare("INSERT INTO investments (user_id, company_id, amount, profit_per_day) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("iidd", $user_id, $company_id, $amount, $profit_per_day);

    if ($stmt->execute()) {
        // Deduct from balance
        $stmt = $conn->prepare("UPDATE users SET balance = balance - ? WHERE id = ?");
        $stmt->bind_param("di", $amount, $user_id);
        $stmt->execute();

        // Record transaction
        record_transaction($user_id, 'investment', $amount, "বিনিয়োগ: " . get_company_name($company_id));

        return ['success' => true, 'message' => 'বিনিয়োগ সফল হয়েছে।'];
    }

    return ['success' => false, 'message' => 'বিনিয়োগ ব্যর্থ হয়েছে।'];
}

function get_company_name($company_id) {
    global $conn;
    $stmt = $conn->prepare("SELECT name FROM companies WHERE id = ?");
    $stmt->bind_param("i", $company_id);
    $stmt->execute();
    $result = $stmt->get_result();
    return $result->fetch_assoc()['name'];
}

// Transaction functions
function record_transaction($user_id, $type, $amount, $description = '', $trx_id = '', $receiver_account = '', $payment_method = '') {
    global $conn;
    $stmt = $conn->prepare("INSERT INTO transactions (user_id, type, amount, description, trx_id, receiver_account, payment_method) VALUES (?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("isdssss", $user_id, $type, $amount, $description, $trx_id, $receiver_account, $payment_method);
    $stmt->execute();
}

function submit_deposit($user_id, $amount, $payment_method, $trx_id) {
    global $conn;

    // Record transaction
    record_transaction($user_id, 'deposit', $amount, "ডিপোজিট: $amount টাকা", $trx_id, '', $payment_method);

    return ['success' => true, 'message' => 'ডিপোজিট রিকুয়েস্ট সাবমিট হয়েছে। অনুমোদনের জন্য অপেক্ষা করুন।'];
}

function submit_withdrawal($user_id, $amount, $payment_method, $receiver_account) {
    global $conn;

    // Check balance
    $user = get_logged_in_user();
    if ($user['balance'] < $amount) {
        return ['success' => false, 'message' => 'যথেষ্ট ব্যালেন্স নেই।'];
    }

    // Record transaction
    record_transaction($user_id, 'withdrawal', $amount, "উইথড্রয়াল: $amount টাকা", '', $receiver_account, $payment_method);

    return ['success' => true, 'message' => 'উইথড্রয়াল রিকুয়েস্ট সাবমিট হয়েছে। অনুমোদনের জন্য অপেক্ষা করুন।'];
}

// Admin functions
function get_pending_transactions() {
    global $conn;
    $stmt = $conn->prepare("SELECT t.*, u.full_name, u.email FROM transactions t JOIN users u ON t.user_id = u.id WHERE t.status = 'pending' ORDER BY t.created_at DESC");
    $stmt->execute();
    return $stmt->get_result();
}

function approve_transaction($transaction_id) {
    global $conn;

    // Get transaction details
    $stmt = $conn->prepare("SELECT * FROM transactions WHERE id = ?");
    $stmt->bind_param("i", $transaction_id);
    $stmt->execute();
    $transaction = $stmt->get_result()->fetch_assoc();

    if ($transaction['type'] == 'deposit') {
        // Add to user balance
        $stmt = $conn->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
        $stmt->bind_param("di", $transaction['amount'], $transaction['user_id']);
        $stmt->execute();
    } elseif ($transaction['type'] == 'withdrawal') {
        // Balance already checked during submission
    }

    // Update transaction status
    $stmt = $conn->prepare("UPDATE transactions SET status = 'approved' WHERE id = ?");
    $stmt->bind_param("i", $transaction_id);
    $stmt->execute();

    return true;
}

function reject_transaction($transaction_id) {
    global $conn;
    $stmt = $conn->prepare("UPDATE transactions SET status = 'rejected' WHERE id = ?");
    $stmt->bind_param("i", $transaction_id);
    $stmt->execute();
    return true;
}

// Get payment methods
function get_payment_methods($type) {
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM payment_methods WHERE type = ? AND is_active = 1");
    $stmt->bind_param("s", $type);
    $stmt->execute();
    return $stmt->get_result();
}

// Get videos
function get_videos($category) {
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM videos WHERE category = ?");
    $stmt->bind_param("s", $category);
    $stmt->execute();
    return $stmt->get_result();
}

// Process daily profits (to be run by cron job)
function process_daily_profits() {
    global $conn;

    $stmt = $conn->prepare("SELECT * FROM investments WHERE status = 'active'");
    $stmt->execute();
    $investments = $stmt->get_result();

    while ($investment = $investments->fetch_assoc()) {
        // Add profit to total_profit
        $stmt = $conn->prepare("UPDATE investments SET total_profit = total_profit + profit_per_day, days_remaining = days_remaining - 1 WHERE id = ?");
        $stmt->bind_param("i", $investment['id']);
        $stmt->execute();

        // Check if investment completed
        $stmt = $conn->prepare("SELECT days_remaining FROM investments WHERE id = ?");
        $stmt->bind_param("i", $investment['id']);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();

        if ($result['days_remaining'] <= 0) {
            // Complete investment and add total profit to user balance
            $stmt = $conn->prepare("UPDATE investments SET status = 'completed' WHERE id = ?");
            $stmt->bind_param("i", $investment['id']);
            $stmt->execute();

            $stmt = $conn->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
            $stmt->bind_param("di", $investment['total_profit'], $investment['user_id']);
            $stmt->execute();

            // Record profit transaction
            record_transaction($investment['user_id'], 'profit', $investment['total_profit'], "বিনিয়োগ লাভ: " . get_company_name($investment['company_id']));
        }
    }
}
?>