// Withdrawal JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // Initialize withdrawal page
    initWithdrawal();
});

function initWithdrawal() {
    // Set up event listeners
    setupWithdrawalEventListeners();

    // Load withdrawal history
    loadWithdrawalHistory();
}

function setupWithdrawalEventListeners() {
    // Quick amount selection
    document.querySelectorAll('.amount-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const amount = this.getAttribute('data-amount');
            document.getElementById('withdrawalAmount').value = amount;

            // Remove active class from all buttons
            document.querySelectorAll('.amount-btn').forEach(b => b.classList.remove('active'));
            // Add active class to clicked button
            this.classList.add('active');
        });
    });

    // Withdrawal form
    const withdrawalForm = document.getElementById('withdrawalForm');
    if (withdrawalForm) {
        withdrawalForm.addEventListener('submit', handleWithdrawal);
    }
}

function handleWithdrawal(e) {
    e.preventDefault();

    const formData = new FormData(e.target);
    const submitBtn = e.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;

    // Validate amount
    const amount = parseFloat(formData.get('amount'));
    const minWithdrawal = parseFloat(document.getElementById('withdrawalAmount').getAttribute('min'));

    if (amount < minWithdrawal) {
        showAlert(`অনুগ্রহ করে কমপক্ষে ${minWithdrawal} টাকা উত্তোলন করুন`, 'danger');
        return;
    }

    // Show loading
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> সাবমিট হচ্ছে...';

    fetch('api/withdrawal.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('উইথড্রয়াল রিকুয়েস্ট সফলভাবে সাবমিট হয়েছে!', 'success');
            e.target.reset();
            loadWithdrawalHistory(); // Reload history
        } else {
            showAlert(data.message || 'উইথড্রয়াল রিকুয়েস্ট ব্যর্থ', 'danger');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('নেটওয়ার্ক ত্রুটি', 'danger');
    })
    .finally(() => {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
    });
}

function loadWithdrawalHistory() {
    fetch('api/get_transactions.php?type=withdrawal')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayWithdrawalHistory(data.transactions);
            }
        })
        .catch(error => {
            console.error('Error loading withdrawal history:', error);
        });
}

function displayWithdrawalHistory(transactions) {
    const container = document.getElementById('withdrawalHistory');

    if (transactions.length === 0) {
        container.innerHTML = '<tr><td colspan="6" class="text-center">কোন উইথড্রয়াল হিস্টরি পাওয়া যায়নি</td></tr>';
        return;
    }

    let html = '';
    transactions.forEach((transaction, index) => {
        const statusBadge = getStatusBadge(transaction.status);
        const date = new Date(transaction.created_at).toLocaleDateString('bn-BD');

        html += `
            <tr>
                <td>${index + 1}</td>
                <td>${date}</td>
                <td>উইথড্রয়াল</td>
                <td>${formatCurrency(transaction.amount)}</td>
                <td>${transaction.receiver_account || '-'}</td>
                <td>${statusBadge}</td>
            </tr>
        `;
    });

    container.innerHTML = html;
}

function getStatusBadge(status) {
    switch (status) {
        case 'approved':
            return '<span class="badge bg-success">অনুমোদিত</span>';
        case 'pending':
            return '<span class="badge bg-warning">অপেক্ষমান</span>';
        case 'rejected':
            return '<span class="badge bg-danger">প্রত্যাখ্যাত</span>';
        default:
            return '<span class="badge bg-secondary">অজানা</span>';
    }
}

function formatCurrency(amount) {
    return new Intl.NumberFormat('bn-BD', {
        style: 'currency',
        currency: 'BDT'
    }).format(amount);
}

function showAlert(message, type = 'info') {
    const alertContainer = document.createElement('div');
    alertContainer.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    alertContainer.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    alertContainer.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    document.body.appendChild(alertContainer);

    // Auto remove after 5 seconds
    setTimeout(() => {
        if (alertContainer.parentNode) {
            alertContainer.remove();
        }
    }, 5000);
}
