// Transaction JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // Initialize transaction page
    initTransaction();
});

function initTransaction() {
    // Load transaction history
    loadTransactionHistory();
}

function loadTransactionHistory() {
    fetch('api/get_transactions.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayTransactionHistory(data.transactions);
            }
        })
        .catch(error => {
            console.error('Error loading transaction history:', error);
        });
}

function displayTransactionHistory(transactions) {
    const container = document.getElementById('transactionHistory');

    if (transactions.length === 0) {
        container.innerHTML = '<tr><td colspan="7" class="text-center">কোন ট্রানজেকশন হিস্টরি নেই</td></tr>';
        return;
    }

    let html = '';
    let runningBalance = 0; // This should ideally be calculated from the database

    transactions.forEach((transaction, index) => {
        const date = new Date(transaction.created_at).toLocaleDateString('bn-BD');
        const debit = transaction.type === 'withdrawal' || transaction.type === 'investment' ? formatCurrency(transaction.amount) : '-';
        const credit = transaction.type === 'deposit' || transaction.type === 'profit' || transaction.type === 'referral_bonus' ? formatCurrency(transaction.amount) : '-';

        // Calculate running balance (simplified)
        if (transaction.type === 'deposit' || transaction.type === 'profit' || transaction.type === 'referral_bonus') {
            runningBalance += parseFloat(transaction.amount);
        } else if (transaction.type === 'withdrawal' || transaction.type === 'investment') {
            runningBalance -= parseFloat(transaction.amount);
        }

        html += `
            <tr>
                <td>${index + 1}</td>
                <td>${date}</td>
                <td>${transaction.description || '-'}</td>
                <td>${transaction.trx_id || transaction.receiver_account || '-'}</td>
                <td class="text-danger">${debit}</td>
                <td class="text-success">${credit}</td>
                <td class="fw-bold">${formatCurrency(runningBalance)}</td>
            </tr>
        `;
    });

    container.innerHTML = html;
}

function formatCurrency(amount) {
    return new Intl.NumberFormat('bn-BD', {
        style: 'currency',
        currency: 'BDT'
    }).format(amount);
}
