// JavaScript for Investment Platform

document.addEventListener('DOMContentLoaded', function() {
    // Initialize the application
    initApp();

    // Initialize scroll animations
    initScrollAnimations();

    // Initialize certificate card interactions
    initCertificateInteractions();
});

function initApp() {
    // Set up event listeners
    setupEventListeners();

    // Check if user is logged in
    checkLoginStatus();
}

function setupEventListeners() {
    // Login form
    const loginForm = document.getElementById('loginForm');
    if (loginForm) {
        loginForm.addEventListener('submit', handleLogin);
    }

    // Register form
    const registerForm = document.getElementById('registerForm');
    if (registerForm) {
        registerForm.addEventListener('submit', handleRegister);
    }

    // Investment form
    const investmentForm = document.getElementById('investmentForm');
    if (investmentForm) {
        investmentForm.addEventListener('submit', handleInvestment);
        // Calculate profit on amount change
        document.getElementById('investmentAmount').addEventListener('input', calculateProfit);
    }

    // Video modal
    const videoModal = document.getElementById('videoModal');
    if (videoModal) {
        videoModal.addEventListener('show.bs.modal', function(event) {
            const button = event.relatedTarget;
            const category = button.getAttribute('data-category');
            loadVideo(category);
        });
    }
}

function checkLoginStatus() {
    // Check if user is logged in and redirect if necessary
    fetch('api/check_login.php')
        .then(response => response.json())
        .then(data => {
            if (data.logged_in && window.location.pathname.includes('index.php')) {
                window.location.href = 'dashboard.php';
            }
        })
        .catch(error => console.error('Error:', error));
}

function loadCompanies(category) {
    fetch(`api/get_companies.php?category=${category}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayCompanies(data.companies, category);
                document.getElementById('companyList').style.display = 'block';
                document.getElementById('companyList').scrollIntoView({ behavior: 'smooth' });
            } else {
                showAlert('কোম্পানি লোড করতে ব্যর্থ হয়েছে', 'danger');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showAlert('নেটওয়ার্ক ত্রুটি', 'danger');
        });
}



function displayCompanies(companies, category) {
    const container = document.getElementById('companiesContainer');
    const title = document.getElementById('categoryTitle');

    title.textContent = category === 'general' ? 'সাধারণ বিনিয়োগ কোম্পানি' : 'মাস্টার বিনিয়োগ কোম্পানি';

    let html = '';
    companies.forEach(company => {
        html += `
            <div class="col-md-6">
                <div class="card company-card h-100">
                    <div class="card-body">
                        <div class="row align-items-center">
                            <div class="col-6">
                                <h5 class="company-name">${company.name}</h5>
                                <p class="return-rate">রিটান: ${company.return_percentage}%</p>
                            </div>
                            <div class="col-6 text-end">
                                <button class="btn btn-primary invest-btn" data-company-id="${company.id}" data-company-name="${company.name}" data-return-rate="${company.return_percentage}">
                                    শেয়ার ক্রয়
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
    });

    container.innerHTML = html;

    // Add event listeners to invest buttons
    document.querySelectorAll('.invest-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            if (!isLoggedIn()) {
                showAlert('বিনিয়োগ করতে আগে লগইন করুন', 'warning');
                const loginModal = new bootstrap.Modal(document.getElementById('loginModal'));
                loginModal.show();
                return;
            }

            const companyId = this.getAttribute('data-company-id');
            const companyName = this.getAttribute('data-company-name');
            const returnRate = this.getAttribute('data-return-rate');

            openInvestmentModal(companyId, companyName, returnRate);
        });
    });
}



function openInvestmentModal(companyId, companyName, returnRate) {
    document.getElementById('investmentDetails').innerHTML = `
        <div class="mb-3">
            <strong>কোম্পানির নাম:</strong> ${companyName}<br>
            <strong>প্রফিটের পার্সেন্ট:</strong> ${returnRate}%
        </div>
    `;
    document.getElementById('companyId').value = companyId;
    document.getElementById('investmentAmount').value = '';
    document.getElementById('totalProfit').value = '';

    const modal = new bootstrap.Modal(document.getElementById('investmentModal'));
    modal.show();
}

function calculateProfit() {
    const amount = parseFloat(document.getElementById('investmentAmount').value) || 0;
    const returnRate = parseFloat(document.querySelector('[data-company-id]').getAttribute('data-return-rate')) || 0;

    const dailyProfit = (amount * returnRate) / 100;
    document.getElementById('totalProfit').value = dailyProfit.toFixed(2) + ' টাকা';
}

function handleLogin(e) {
    e.preventDefault();

    const formData = new FormData(e.target);
    const submitBtn = e.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;

    // Show loading
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> লগইন হচ্ছে...';

    fetch('api/login.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('লগইন সফল!', 'success');
            setTimeout(() => {
                window.location.href = 'dashboard.php';
            }, 1000);
        } else {
            showAlert(data.message || 'লগইন ব্যর্থ', 'danger');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('নেটওয়ার্ক ত্রুটি', 'danger');
    })
    .finally(() => {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
    });
}

function handleRegister(e) {
    e.preventDefault();

    const formData = new FormData(e.target);

    // Validate passwords
    if (formData.get('password') !== formData.get('confirm_password')) {
        showAlert('পাসওয়ার্ড মিলছে না', 'danger');
        return;
    }

    const submitBtn = e.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;

    // Show loading
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> রেজিস্টার হচ্ছে...';

    fetch('api/register.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('রেজিস্ট্রেশন সফল! লগইন করুন', 'success');
            bootstrap.Modal.getInstance(document.getElementById('registerModal')).hide();
            setTimeout(() => {
                const loginModal = new bootstrap.Modal(document.getElementById('loginModal'));
                loginModal.show();
            }, 1500);
        } else {
            showAlert(data.message || 'রেজিস্ট্রেশন ব্যর্থ', 'danger');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('নেটওয়ার্ক ত্রুটি', 'danger');
    })
    .finally(() => {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
    });
}

function handleInvestment(e) {
    e.preventDefault();

    if (!confirm('আপনি কি নিশ্চিত যে বিনিয়োগ করতে চান?')) {
        return;
    }

    const formData = new FormData(e.target);
    const submitBtn = e.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;

    // Show loading
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> বিনিয়োগ হচ্ছে...';

    fetch('api/invest.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('বিনিয়োগ সফল!', 'success');
            bootstrap.Modal.getInstance(document.getElementById('investmentModal')).hide();
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        } else {
            showAlert(data.message || 'বিনিয়োগ ব্যর্থ', 'danger');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('নেটওয়ার্ক ত্রুটি', 'danger');
    })
    .finally(() => {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
    });
}

function loadVideo(category) {
    fetch(`api/get_video.php?category=${category}`)
        .then(response => response.json())
        .then(data => {
            if (data.success && data.video) {
                document.getElementById('videoTitle').textContent = data.video.title;
                document.getElementById('videoFrame').src = data.video.url;
            } else {
                document.getElementById('videoTitle').textContent = 'ভিডিও পাওয়া যায়নি';
                document.getElementById('videoFrame').src = '';
            }
        })
        .catch(error => {
            console.error('Error:', error);
            document.getElementById('videoTitle').textContent = 'ভিডিও লোড করতে ব্যর্থ';
        });
}



function isLoggedIn() {
    // Simple check - in real app, check session
    return document.cookie.includes('user_id') || localStorage.getItem('logged_in') === 'true';
}

function showAlert(message, type = 'info') {
    const alertContainer = document.createElement('div');
    alertContainer.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    alertContainer.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    alertContainer.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    document.body.appendChild(alertContainer);

    // Auto remove after 5 seconds
    setTimeout(() => {
        if (alertContainer.parentNode) {
            alertContainer.remove();
        }
    }, 5000);
}

// Utility functions for dashboard
function formatCurrency(amount) {
    return new Intl.NumberFormat('bn-BD', {
        style: 'currency',
        currency: 'BDT'
    }).format(amount);
}

function updateBalance(balance) {
    const balanceElements = document.querySelectorAll('.balance-amount');
    balanceElements.forEach(el => {
        el.textContent = formatCurrency(balance);
    });
}

// Dashboard specific functions
function loadDashboardData() {
    fetch('api/dashboard_data.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateBalance(data.balance);
                // Update other dashboard elements
            }
        })
        .catch(error => console.error('Error loading dashboard data:', error));
}

// Deposit/Withdrawal functions
function submitDeposit(amount, method, trxId) {
    const formData = new FormData();
    formData.append('amount', amount);
    formData.append('payment_method', method);
    formData.append('trx_id', trxId);

    return fetch('api/deposit.php', {
        method: 'POST',
        body: formData
    }).then(response => response.json());
}

function submitWithdrawal(amount, method, receiverAccount) {
    const formData = new FormData();
    formData.append('amount', amount);
    formData.append('payment_method', method);
    formData.append('receiver_account', receiverAccount);

    return fetch('api/withdrawal.php', {
        method: 'POST',
        body: formData
    }).then(response => response.json());
}

// Admin functions
function approveTransaction(transactionId) {
    return fetch(`api/admin/approve_transaction.php?id=${transactionId}`)
        .then(response => response.json());
}

function rejectTransaction(transactionId) {
    return fetch(`api/admin/reject_transaction.php?id=${transactionId}`)
        .then(response => response.json());
}

// Scroll Animations and Certificate Interactions
function initScrollAnimations() {
    // Intersection Observer for scroll animations
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-in');
            }
        });
    }, observerOptions);

    // Observe certificate cards
    document.querySelectorAll('.certificate-card').forEach(card => {
        observer.observe(card);
    });

    // Observe other elements that should animate on scroll
    document.querySelectorAll('.animate-on-scroll').forEach(element => {
        observer.observe(element);
    });
}

function initCertificateInteractions() {
    // Certificate card click effects
    document.querySelectorAll('.certificate-card').forEach(card => {
        card.addEventListener('click', function(e) {
            // Don't trigger if clicking on PDF link
            if (e.target.closest('.btn')) return;

            // Add click animation
            this.style.transform = 'scale(0.98)';
            setTimeout(() => {
                this.style.transform = '';
            }, 150);
        });

        // Certificate image zoom on hover
        const image = card.querySelector('.certificate-image');
        if (image) {
            image.addEventListener('mouseenter', function() {
                this.style.transform = 'scale(1.1)';
                this.style.zIndex = '10';
            });

            image.addEventListener('mouseleave', function() {
                this.style.transform = 'scale(1)';
                this.style.zIndex = '';
            });
        }
    });

    // Smooth scrolling for certificate section
    const certificateSection = document.querySelector('.certificates-section');
    if (certificateSection) {
        // Add parallax effect on scroll
        window.addEventListener('scroll', function() {
            const scrolled = window.pageYOffset;
            const rate = scrolled * -0.5;

            if (certificateSection.offsetTop < scrolled + window.innerHeight &&
                certificateSection.offsetTop + certificateSection.offsetHeight > scrolled) {
                certificateSection.style.transform = `translateY(${rate * 0.1}px)`;
            }
        });
    }

    // Certificate modal for larger view
    initCertificateModal();
}

function initCertificateModal() {
    // Create modal for certificate viewing
    if (!document.getElementById('certificateModal')) {
        const modalHtml = `
            <div class="modal fade" id="certificateModal" tabindex="-1">
                <div class="modal-dialog modal-lg modal-dialog-centered">
                    <div class="modal-content" style="background: var(--background-card); border: 2px solid var(--border-teal);">
                        <div class="modal-header" style="background: var(--primary-teal); border-bottom: 1px solid var(--border-teal);">
                            <h5 class="modal-title text-white" id="certificateModalTitle">সার্টিফিকেট দেখুন</h5>
                            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body text-center">
                            <img id="certificateModalImage" src="" alt="Certificate" class="img-fluid" style="max-height: 70vh; border-radius: 10px; box-shadow: 0 5px 15px rgba(0,0,0,0.3);">
                        </div>
                    </div>
                </div>
            </div>
        `;
        document.body.insertAdjacentHTML('beforeend', modalHtml);
    }

    // Add click event to certificate images
    document.querySelectorAll('.certificate-image').forEach(img => {
        img.addEventListener('click', function() {
            const src = this.src;
            const title = this.closest('.certificate-card').querySelector('.certificate-title').textContent;

            document.getElementById('certificateModalImage').src = src;
            document.getElementById('certificateModalTitle').textContent = title;

            const modal = new bootstrap.Modal(document.getElementById('certificateModal'));
            modal.show();
        });
    });
}

// Add CSS animation class dynamically
const style = document.createElement('style');
style.textContent = `
    .animate-in {
        animation: slideInUp 0.6s ease-out forwards !important;
    }

    @keyframes slideInUp {
        from {
            opacity: 0;
            transform: translateY(30px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
`;
document.head.appendChild(style);
