// History JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // Initialize history page
    initHistory();
});

function initHistory() {
    // Load active investments
    loadActiveInvestments();

    // Load investment history
    loadInvestmentHistory();
}

function loadActiveInvestments() {
    fetch('api/get_investments.php?status=active')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayActiveInvestments(data.investments);
            }
        })
        .catch(error => {
            console.error('Error loading active investments:', error);
        });
}

function loadInvestmentHistory() {
    fetch('api/get_investments.php?status=all')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayInvestmentHistory(data.investments);
            }
        })
        .catch(error => {
            console.error('Error loading investment history:', error);
        });
}

function displayActiveInvestments(investments) {
    const container = document.getElementById('activeInvestments');

    if (investments.length === 0) {
        container.innerHTML = '<tr><td colspan="7" class="text-center">কোন অ্যাক্টিভ বিনিয়োগ নেই</td></tr>';
        return;
    }

    let html = '';
    investments.forEach((investment, index) => {
        const daysLeft = investment.days_remaining;
        const statusBadge = getInvestmentStatusBadge(investment.status);

        html += `
            <tr>
                <td>${index + 1}</td>
                <td>${investment.company_name}</td>
                <td>${investment.created_at.split(' ')[0]}</td>
                <td>${formatCurrency(investment.amount)}</td>
                <td>${formatCurrency(investment.profit_per_day)}</td>
                <td>${daysLeft} দিন</td>
                <td>${statusBadge}</td>
            </tr>
        `;
    });

    container.innerHTML = html;
}

function displayInvestmentHistory(investments) {
    const container = document.getElementById('investmentHistory');

    if (investments.length === 0) {
        container.innerHTML = '<tr><td colspan="7" class="text-center">কোন বিনিয়োগ হিস্টরি নেই</td></tr>';
        return;
    }

    let html = '';
    investments.forEach((investment, index) => {
        const startDate = new Date(investment.created_at).toLocaleDateString('bn-BD');
        const statusBadge = getInvestmentStatusBadge(investment.status);

        html += `
            <tr>
                <td>${index + 1}</td>
                <td>${investment.company_name}</td>
                <td>${formatCurrency(investment.amount)}</td>
                <td>${formatCurrency(investment.profit_per_day)}</td>
                <td>${formatCurrency(investment.total_profit)}</td>
                <td>${startDate}</td>
                <td>${statusBadge}</td>
            </tr>
        `;
    });

    container.innerHTML = html;
}

function getInvestmentStatusBadge(status) {
    switch (status) {
        case 'active':
            return '<span class="badge bg-success">চলমান</span>';
        case 'completed':
            return '<span class="badge bg-primary">সম্পন্ন</span>';
        case 'cancelled':
            return '<span class="badge bg-danger">বাতিল</span>';
        default:
            return '<span class="badge bg-secondary">অজানা</span>';
    }
}

function formatCurrency(amount) {
    return new Intl.NumberFormat('bn-BD', {
        style: 'currency',
        currency: 'BDT'
    }).format(amount);
}
