// Dashboard JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // Initialize dashboard
    initDashboard();
});

function initDashboard() {
    // Set up event listeners for dashboard/sidebar
    setupDashboardEventListeners();

    // Load initial data if on dashboard page (check for dashboard-specific elements)
    if (document.getElementById('companiesContainerDashboard')) {
        loadDashboardData();
        loadNotices(); // Load notices for dashboard
    }
}

function setupDashboardEventListeners() {
    // Sidebar toggle functionality
    const sidebarToggle = document.getElementById('sidebarToggle');
    const sidebar = document.getElementById('sidebar');
    const backdrop = document.getElementById('navbarBackdrop');

    if (sidebarToggle && sidebar && backdrop) {
        sidebarToggle.addEventListener('click', function() {
            toggleSidebar();
        });

        backdrop.addEventListener('click', function() {
            closeSidebar();
        });

        // Close sidebar when clicking on nav links
        sidebar.querySelectorAll('.nav-link').forEach(link => {
            link.addEventListener('click', function() {
                closeSidebar();
            });
        });

        // Close sidebar on window resize if screen becomes larger
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                closeSidebar();
            }
        });
    }

    // Category selection for dashboard
    document.querySelectorAll('.category-btn-dashboard').forEach(btn => {
        btn.addEventListener('click', function() {
            const category = this.getAttribute('data-category');
            loadCompaniesDashboard(category);
        });
    });

    // Investment form
    const investmentForm = document.getElementById('investmentForm');
    if (investmentForm) {
        investmentForm.addEventListener('submit', handleInvestment);
        // Calculate profit on amount change
        document.getElementById('investmentAmount').addEventListener('input', calculateProfit);
    }
}

function loadCompaniesDashboard(category) {
    fetch(`api/get_companies.php?category=${category}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayCompaniesDashboard(data.companies, category);
                const companyList = document.getElementById('companyListDashboard');
                if (companyList) {
                    companyList.style.display = 'block';
                    companyList.scrollIntoView({ behavior: 'smooth' });
                }
            } else {
                showAlert('কোম্পানি লোড করতে ব্যর্থ হয়েছে', 'danger');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showAlert('নেটওয়ার্ক ত্রুটি', 'danger');
        });
}

function displayCompaniesDashboard(companies, category) {
    const container = document.getElementById('companiesContainerDashboard');

    // Check if container exists (page might have changed during async operation)
    if (!container) {
        return;
    }

    let html = '';
    companies.forEach(company => {
        html += `
            <tr>
                <td class="text-start company-name">${company.name}</td>
                <td class="text-center return-rate"><strong>${company.return_percentage}%</strong></td>
                <td class="text-end">
                    <button class="btn btn-share invest-btn-dashboard" data-company-id="${company.id}" data-company-name="${company.name}" data-return-rate="${company.return_percentage}">
                        শেয়ার ক্রয়
                    </button>
                </td>
            </tr>
        `;
    });

    container.innerHTML = html;

    // Add event listeners to invest buttons
    document.querySelectorAll('.invest-btn-dashboard').forEach(btn => {
        btn.addEventListener('click', function() {
            const companyId = this.getAttribute('data-company-id');
            const companyName = this.getAttribute('data-company-name');
            const returnRate = this.getAttribute('data-return-rate');

            openInvestmentModal(companyId, companyName, returnRate, this);
        });
    });
}

let investmentModalTrigger = null;

function openInvestmentModal(companyId, companyName, returnRate, triggerButton = null) {
    // Store the triggering button for focus management
    investmentModalTrigger = triggerButton;

    document.getElementById('companyNameBanner').textContent = companyName;
    document.getElementById('companyNameDetail').textContent = companyName;
    document.getElementById('returnRateDetail').textContent = returnRate + '%';
    document.getElementById('companyId').value = companyId;
    document.getElementById('investmentAmount').value = '';
    document.getElementById('totalProfitDisplay').textContent = '৳ 0 (প্রতি ১ দিন এ)';

    const modalElement = document.getElementById('investmentModal');
    const modal = new bootstrap.Modal(modalElement, {
        backdrop: 'static', // Prevent closing by clicking backdrop
        keyboard: true, // Allow closing with ESC key
        focus: true
    });

    // Ensure smooth animation by adding a small delay
    setTimeout(() => {
        modal.show();
    }, 10);

    // Add event listener for when modal is hidden to manage focus
    modalElement.addEventListener('hidden.bs.modal', function() {
        // Return focus to the triggering button to prevent accessibility violation
        if (investmentModalTrigger && investmentModalTrigger.focus) {
            investmentModalTrigger.focus();
        }
        // Reset the trigger
        investmentModalTrigger = null;
    }, { once: true }); // Use once to avoid multiple listeners
}

function calculateProfit() {
    const amount = parseFloat(document.getElementById('investmentAmount').value) || 0;
    const returnRate = parseFloat(document.querySelector('[data-company-id]').getAttribute('data-return-rate')) || 0;

    const dailyProfit = (amount * returnRate) / 100;
    document.getElementById('totalProfitDisplay').textContent = '৳ ' + dailyProfit.toFixed(2) + ' (প্রতি ১ দিন এ)';
}

function handleInvestment(e) {
    e.preventDefault();

    if (!confirm('আপনি কি নিশ্চিত যে বিনিয়োগ করতে চান?')) {
        return;
    }

    const formData = new FormData(e.target);
    const submitBtn = e.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;

    // Show loading
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> বিনিয়োগ হচ্ছে...';

    fetch('api/invest.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('বিনিয়োগ সফল!', 'success');
            investmentModalTrigger = null; // Clear trigger since page will reload
            bootstrap.Modal.getInstance(document.getElementById('investmentModal')).hide();
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        } else {
            showAlert(data.message || 'বিনিয়োগ ব্যর্থ', 'danger');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('নেটওয়ার্ক ত্রুটি', 'danger');
    })
    .finally(() => {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
    });
}

function loadDashboardData() {
    // Load all companies by default
    loadCompaniesDashboard('');
}

function showAlert(message, type = 'info') {
    const alertContainer = document.createElement('div');
    alertContainer.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    alertContainer.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    alertContainer.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    document.body.appendChild(alertContainer);

    // Auto remove after 5 seconds
    setTimeout(() => {
        if (alertContainer.parentNode) {
            alertContainer.remove();
        }
    }, 5000);
}

// Sidebar functions
function toggleSidebar() {
    const sidebar = document.getElementById('sidebar');
    const backdrop = document.getElementById('navbarBackdrop');
    const body = document.body;

    if (sidebar.classList.contains('show')) {
        closeSidebar();
    } else {
        openSidebar();
    }
}

function openSidebar() {
    const sidebar = document.getElementById('sidebar');
    const backdrop = document.getElementById('navbarBackdrop');
    const body = document.body;

    sidebar.classList.add('show');
    backdrop.classList.add('show');
    body.classList.add('sidebar-open');

    // Prevent body scroll when sidebar is open
    body.style.overflow = 'hidden';
}

function closeSidebar() {
    const sidebar = document.getElementById('sidebar');
    const backdrop = document.getElementById('navbarBackdrop');
    const body = document.body;

    sidebar.classList.remove('show');
    backdrop.classList.remove('show');
    body.classList.remove('sidebar-open');

    // Restore body scroll
    body.style.overflow = '';
}

// Notice functions
let currentNoticeIndex = 0;
let notices = [];
let noticeInterval = null;

function loadNotices() {
    fetch('api/get_notices.php')
        .then(response => response.json())
        .then(data => {
            if (data.success && data.notices.length > 0) {
                notices = data.notices;
                displayCurrentNotice();
                startNoticeRotation();
                document.getElementById('noticeSection').style.display = 'block';
            }
        })
        .catch(error => {
            console.error('Error loading notices:', error);
        });
}

function displayCurrentNotice() {
    if (notices.length === 0) return;

    const notice = notices[currentNoticeIndex];
    const noticeSection = document.getElementById('noticeSection');
    const noticeAlert = noticeSection.querySelector('.notice-alert');

    // Remove previous animation class
    noticeAlert.classList.remove('slide-in-notice');
    void noticeAlert.offsetWidth; // Trigger reflow

    // Update content
    document.getElementById('noticeTitle').innerHTML = `<i class="fas fa-bullhorn"></i> ${notice.title}`;
    document.getElementById('noticeMessage').textContent = notice.message;

    // Add animation class
    noticeAlert.classList.add('slide-in-notice');
}

function startNoticeRotation() {
    if (notices.length <= 1) return; // No need to rotate if only one notice

    noticeInterval = setInterval(() => {
        currentNoticeIndex = (currentNoticeIndex + 1) % notices.length;
        displayCurrentNotice();
    }, 8000); // Change notice every 8 seconds
}

function hideNotice() {
    const noticeSection = document.getElementById('noticeSection');
    noticeSection.style.display = 'none';

    // Stop rotation if running
    if (noticeInterval) {
        clearInterval(noticeInterval);
        noticeInterval = null;
    }
}

// Utility functions
function formatCurrency(amount) {
    return new Intl.NumberFormat('bn-BD', {
        style: 'currency',
        currency: 'BDT'
    }).format(amount);
}

function updateBalance(balance) {
    const balanceElements = document.querySelectorAll('.balance-amount');
    balanceElements.forEach(el => {
        el.textContent = formatCurrency(balance);
    });
}
