<?php
require_once '../includes/config.php';

// Simple admin check
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: index.php");
    exit();
}

// Handle user actions
if (isset($_GET['action']) && isset($_GET['user_id'])) {
    $user_id = (int)$_GET['user_id'];

    if ($_GET['action'] === 'activate') {
        $stmt = $conn->prepare("UPDATE users SET status = 'active' WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $message = "ইউজার অ্যাক্টিভেট হয়েছে";
    } elseif ($_GET['action'] === 'deactivate') {
        $stmt = $conn->prepare("UPDATE users SET status = 'inactive' WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $message = "ইউজার ডিঅ্যাক্টিভেট হয়েছে";
    } elseif ($_GET['action'] === 'ban') {
        $stmt = $conn->prepare("UPDATE users SET status = 'banned' WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $message = "ইউজার ব্যান হয়েছে";
    } elseif ($_GET['action'] === 'unban') {
        $stmt = $conn->prepare("UPDATE users SET status = 'active' WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $message = "ইউজার আনব্যান হয়েছে";
    }
}

// Handle balance actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'add_balance') {
        $user_id = (int)$_POST['user_id'];
        $amount = (float)$_POST['amount'];
        $reason = trim($_POST['reason']);

        if ($amount > 0) {
            $stmt = $conn->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
            $stmt->bind_param("di", $amount, $user_id);
            $stmt->execute();

            // Record transaction
            record_transaction($user_id, 'admin_credit', $amount, "অ্যাডমিন ব্যালেন্স যোগ: " . $reason);

            $message = "ব্যালেন্স সফলভাবে যোগ হয়েছে";
        }
    } elseif ($_POST['action'] === 'remove_balance') {
        $user_id = (int)$_POST['user_id'];
        $amount = (float)$_POST['amount'];
        $reason = trim($_POST['reason']);

        if ($amount > 0) {
            // Check current balance
            $stmt = $conn->prepare("SELECT balance FROM users WHERE id = ?");
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $current_balance = $stmt->get_result()->fetch_assoc()['balance'];

            if ($current_balance >= $amount) {
                $stmt = $conn->prepare("UPDATE users SET balance = balance - ? WHERE id = ?");
                $stmt->bind_param("di", $amount, $user_id);
                $stmt->execute();

                // Record transaction
                record_transaction($user_id, 'admin_debit', $amount, "অ্যাডমিন ব্যালেন্স কাটা: " . $reason);

                $message = "ব্যালেন্স সফলভাবে কাটা হয়েছে";
            } else {
                $message = "যথেষ্ট ব্যালেন্স নেই";
            }
        }
    }
}

// Get users with pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 20;
$offset = ($page - 1) * $limit;

$total_users = $conn->query("SELECT COUNT(*) as count FROM users")->fetch_assoc()['count'];
$total_pages = ceil($total_users / $limit);

$stmt = $conn->prepare("SELECT * FROM users ORDER BY created_at DESC LIMIT ? OFFSET ?");
$stmt->bind_param("ii", $limit, $offset);
$stmt->execute();
$users = $stmt->get_result();
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ইউজার ম্যানেজমেন্ট - <?php echo $company_name; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include '../includes/admin_navbar.php'; ?>

    <div class="container mt-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2><i class="fas fa-users"></i> ইউজার ম্যানেজমেন্ট</h2>
            <div>
                <span class="badge bg-primary fs-6">মোট ইউজার: <?php echo $total_users; ?></span>
            </div>
        </div>

        <?php if (isset($message)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?php echo $message; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <!-- Search and Filter -->
        <div class="card mb-4">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-4">
                        <input type="text" class="form-control" id="searchUser" placeholder="ইউজার খুঁজুন...">
                    </div>
                    <div class="col-md-3">
                        <select class="form-control" id="statusFilter">
                            <option value="">সকল স্ট্যাটাস</option>
                            <option value="active">অ্যাক্টিভ</option>
                            <option value="inactive">ইনঅ্যাক্টিভ</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <button class="btn btn-primary w-100" onclick="filterUsers()">
                            <i class="fas fa-search"></i> খুঁজুন
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Users Table -->
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>আইডি</th>
                                <th>নাম</th>
                                <th>ইমেইল</th>
                                <th>ফোন</th>
                                <th>ব্যালেন্স</th>
                                <th>স্ট্যাটাস</th>
                                <th>যোগদানের তারিখ</th>
                                <th>অ্যাকশন</th>
                            </tr>
                        </thead>
                        <tbody id="usersTable">
                            <?php while ($user = $users->fetch_assoc()): ?>
                            <tr>
                                <td><?php echo $user['id']; ?></td>
                                <td><?php echo htmlspecialchars($user['full_name']); ?></td>
                                <td><?php echo htmlspecialchars($user['email']); ?></td>
                                <td><?php echo htmlspecialchars($user['phone']); ?></td>
                                <td><?php echo format_currency($user['balance']); ?></td>
                                <td>
                                    <?php
                                    $status_class = 'secondary';
                                    $status_text = 'ইনঅ্যাক্টিভ';
                                    if (isset($user['status'])) {
                                        switch ($user['status']) {
                                            case 'active':
                                                $status_class = 'success';
                                                $status_text = 'অ্যাক্টিভ';
                                                break;
                                            case 'inactive':
                                                $status_class = 'warning';
                                                $status_text = 'ইনঅ্যাক্টিভ';
                                                break;
                                            case 'banned':
                                                $status_class = 'danger';
                                                $status_text = 'ব্যান';
                                                break;
                                        }
                                    }
                                    ?>
                                    <span class="badge bg-<?php echo $status_class; ?>">
                                        <?php echo $status_text; ?>
                                    </span>
                                </td>
                                <td><?php echo date('d/m/Y H:i', strtotime($user['created_at'])); ?></td>
                                <td>
                                    <div class="btn-group" role="group">
                                        <a href="user_details.php?id=<?php echo $user['id']; ?>" class="btn btn-sm btn-info" title="বিস্তারিত দেখুন">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <button class="btn btn-sm btn-primary" onclick="showBalanceModal(<?php echo $user['id']; ?>, '<?php echo addslashes($user['full_name']); ?>', <?php echo $user['balance']; ?>)" title="ব্যালেন্স ম্যানেজ">
                                            <i class="fas fa-wallet"></i>
                                        </button>
                                        <?php if (!isset($user['status']) || $user['status'] === 'inactive'): ?>
                                        <a href="?action=activate&user_id=<?php echo $user['id']; ?>" class="btn btn-sm btn-success" onclick="return confirm('এই ইউজারকে অ্যাক্টিভেট করতে চান?')" title="অ্যাক্টিভেট">
                                            <i class="fas fa-check"></i>
                                        </a>
                                        <?php elseif ($user['status'] === 'active'): ?>
                                        <a href="?action=ban&user_id=<?php echo $user['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('এই ইউজারকে ব্যান করতে চান?')" title="ব্যান">
                                            <i class="fas fa-ban"></i>
                                        </a>
                                        <?php elseif ($user['status'] === 'banned'): ?>
                                        <a href="?action=unban&user_id=<?php echo $user['id']; ?>" class="btn btn-sm btn-warning" onclick="return confirm('এই ইউজারকে আনব্যান করতে চান?')" title="আনব্যান">
                                            <i class="fas fa-undo"></i>
                                        </a>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                <nav aria-label="User pagination" class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php if ($page > 1): ?>
                        <li class="page-item">
                            <a class="page-link" href="?page=<?php echo $page - 1; ?>">পূর্ববর্তী</a>
                        </li>
                        <?php endif; ?>

                        <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                        <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                        </li>
                        <?php endfor; ?>

                        <?php if ($page < $total_pages): ?>
                        <li class="page-item">
                            <a class="page-link" href="?page=<?php echo $page + 1; ?>">পরবর্তী</a>
                        </li>
                        <?php endif; ?>
                    </ul>
                </nav>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Balance Management Modal -->
    <div class="modal fade" id="balanceModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">ব্যালেন্স ম্যানেজমেন্ট - <span id="userName"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">বর্তমান ব্যালেন্স: <span id="currentBalance" class="fw-bold"></span></label>
                    </div>

                    <!-- Add Balance Tab -->
                    <ul class="nav nav-tabs" id="balanceTab" role="tablist">
                        <li class="nav-item" role="presentation">
                            <button class="nav-link active" id="add-tab" data-bs-toggle="tab" data-bs-target="#addBalance" type="button" role="tab">ব্যালেন্স যোগ</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="remove-tab" data-bs-toggle="tab" data-bs-target="#removeBalance" type="button" role="tab">ব্যালেন্স কাটা</button>
                        </li>
                    </ul>

                    <div class="tab-content mt-3" id="balanceTabContent">
                        <!-- Add Balance -->
                        <div class="tab-pane fade show active" id="addBalance" role="tabpanel">
                            <form id="addBalanceForm" method="POST">
                                <input type="hidden" name="action" value="add_balance">
                                <input type="hidden" name="user_id" id="addUserId">
                                <div class="mb-3">
                                    <label class="form-label">পরিমান (টাকা)</label>
                                    <input type="number" class="form-control" name="amount" min="0.01" step="0.01" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">কারণ</label>
                                    <textarea class="form-control" name="reason" rows="2" placeholder="ব্যালেন্স যোগের কারণ লিখুন..." required></textarea>
                                </div>
                                <button type="submit" class="btn btn-success w-100">
                                    <i class="fas fa-plus"></i> ব্যালেন্স যোগ করুন
                                </button>
                            </form>
                        </div>

                        <!-- Remove Balance -->
                        <div class="tab-pane fade" id="removeBalance" role="tabpanel">
                            <form id="removeBalanceForm" method="POST">
                                <input type="hidden" name="action" value="remove_balance">
                                <input type="hidden" name="user_id" id="removeUserId">
                                <div class="mb-3">
                                    <label class="form-label">পরিমান (টাকা)</label>
                                    <input type="number" class="form-control" name="amount" min="0.01" step="0.01" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">কারণ</label>
                                    <textarea class="form-control" name="reason" rows="2" placeholder="ব্যালেন্স কাটার কারণ লিখুন..." required></textarea>
                                </div>
                                <button type="submit" class="btn btn-danger w-100">
                                    <i class="fas fa-minus"></i> ব্যালেন্স কাটুন
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function showBalanceModal(userId, userName, currentBalance) {
            document.getElementById('userName').textContent = userName;
            document.getElementById('currentBalance').textContent = new Intl.NumberFormat('bn-BD', {
                style: 'currency',
                currency: 'BDT'
            }).format(currentBalance);
            document.getElementById('addUserId').value = userId;
            document.getElementById('removeUserId').value = userId;

            const modal = new bootstrap.Modal(document.getElementById('balanceModal'));
            modal.show();
        }

        // Auto-submit forms and reload page
        document.getElementById('addBalanceForm').addEventListener('submit', function(e) {
            // Form will submit normally, page will reload with message
        });

        document.getElementById('removeBalanceForm').addEventListener('submit', function(e) {
            // Form will submit normally, page will reload with message
        });
    </script>
    <script src="../assets/js/admin_users.js"></script>
</body>
</html>
