<?php
require_once '../includes/config.php';

// Simple admin check
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: index.php");
    exit();
}

if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header("Location: users.php");
    exit();
}

$user_id = (int)$_GET['id'];

// Get user details
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$user = $stmt->get_result()->fetch_assoc();

if (!$user) {
    header("Location: users.php");
    exit();
}

// Get user's transactions
$stmt = $conn->prepare("SELECT * FROM transactions WHERE user_id = ? ORDER BY created_at DESC LIMIT 20");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$transactions = $stmt->get_result();

// Get user's investments
$stmt = $conn->prepare("SELECT i.*, c.name as company_name FROM investments i JOIN companies c ON i.company_id = c.id WHERE i.user_id = ? ORDER BY i.created_at DESC LIMIT 10");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$investments = $stmt->get_result();

// Get referral statistics
$stmt = $conn->prepare("SELECT COUNT(*) as total_referred FROM users WHERE referrer_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$referral_count = $stmt->get_result()->fetch_assoc()['total_referred'];

$stmt = $conn->prepare("SELECT SUM(commission) as total_commission FROM referral_history WHERE user_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$total_commission = $stmt->get_result()->fetch_assoc()['total_commission'] ?: 0;
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ইউজার বিস্তারিত - <?php echo htmlspecialchars($user['full_name']); ?> - <?php echo $company_name; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include '../includes/admin_navbar.php'; ?>

    <div class="container mt-4">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="fas fa-user"></i> ইউজার বিস্তারিত</h2>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb">
                        <li class="breadcrumb-item"><a href="dashboard.php">ড্যাশবোর্ড</a></li>
                        <li class="breadcrumb-item"><a href="users.php">ইউজারস</a></li>
                        <li class="breadcrumb-item active"><?php echo htmlspecialchars($user['full_name']); ?></li>
                    </ol>
                </nav>
            </div>
            <a href="users.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> ফিরে যান
            </a>
        </div>

        <!-- User Info Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4><?php echo htmlspecialchars($user['full_name']); ?></h4>
                        <p class="text-muted">ইউজার আইডি: <?php echo $user['id']; ?></p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-primary"><?php echo format_currency($user['balance']); ?></h4>
                        <p>বর্তমান ব্যালেন্স</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-success"><?php echo $referral_count; ?></h4>
                        <p>রেফারেল কাউন্ট</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-info"><?php echo format_currency($total_commission); ?></h4>
                        <p>মোট কমিশন</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- User Details -->
        <div class="row mb-4">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-info-circle"></i> ব্যক্তিগত তথ্য</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <p><strong>নাম:</strong> <?php echo htmlspecialchars($user['full_name']); ?></p>
                                <p><strong>ইমেইল:</strong> <?php echo htmlspecialchars($user['email']); ?></p>
                                <p><strong>ফোন:</strong> <?php echo htmlspecialchars($user['phone']); ?></p>
                            </div>
                            <div class="col-md-6">
                                <p><strong>রেফারেল কোড:</strong> <?php echo htmlspecialchars($user['referral_code']); ?></p>
                                <p><strong>স্ট্যাটাস:</strong>
                                    <?php
                                    $status_class = 'secondary';
                                    $status_text = 'ইনঅ্যাক্টিভ';
                                    if (isset($user['status'])) {
                                        switch ($user['status']) {
                                            case 'active':
                                                $status_class = 'success';
                                                $status_text = 'অ্যাক্টিভ';
                                                break;
                                            case 'inactive':
                                                $status_class = 'warning';
                                                $status_text = 'ইনঅ্যাক্টিভ';
                                                break;
                                            case 'banned':
                                                $status_class = 'danger';
                                                $status_text = 'ব্যান';
                                                break;
                                        }
                                    }
                                    ?>
                                    <span class="badge bg-<?php echo $status_class; ?>"><?php echo $status_text; ?></span>
                                </p>
                                <p><strong>যোগদানের তারিখ:</strong> <?php echo date('d/m/Y H:i', strtotime($user['created_at'])); ?></p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-cogs"></i> অ্যাকশন</h5>
                    </div>
                    <div class="card-body">
                        <button class="btn btn-primary w-100 mb-2" onclick="showBalanceModal(<?php echo $user['id']; ?>, '<?php echo addslashes($user['full_name']); ?>', <?php echo $user['balance']; ?>)">
                            <i class="fas fa-wallet"></i> ব্যালেন্স ম্যানেজ
                        </button>
                        <?php if (!isset($user['status']) || $user['status'] === 'inactive'): ?>
                        <a href="users.php?action=activate&user_id=<?php echo $user['id']; ?>" class="btn btn-success w-100 mb-2" onclick="return confirm('এই ইউজারকে অ্যাক্টিভেট করতে চান?')">
                            <i class="fas fa-check"></i> অ্যাক্টিভেট
                        </a>
                        <?php elseif ($user['status'] === 'active'): ?>
                        <a href="users.php?action=ban&user_id=<?php echo $user['id']; ?>" class="btn btn-danger w-100 mb-2" onclick="return confirm('এই ইউজারকে ব্যান করতে চান?')">
                            <i class="fas fa-ban"></i> ব্যান
                        </a>
                        <?php elseif ($user['status'] === 'banned'): ?>
                        <a href="users.php?action=unban&user_id=<?php echo $user['id']; ?>" class="btn btn-warning w-100 mb-2" onclick="return confirm('এই ইউজারকে আনব্যান করতে চান?')">
                            <i class="fas fa-undo"></i> আনব্যান
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Transactions -->
        <div class="card mb-4">
            <div class="card-header">
                <h5><i class="fas fa-exchange-alt"></i> সাম্প্রতিক ট্রানজেকশন (সর্বশেষ ২০টি)</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>টাইপ</th>
                                <th>পরিমান</th>
                                <th>স্ট্যাটাস</th>
                                <th>তারিখ</th>
                                <th>বিস্তারিত</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($transactions->num_rows > 0): ?>
                                <?php while ($transaction = $transactions->fetch_assoc()): ?>
                                <tr>
                                    <td>
                                        <?php
                                        $type_labels = [
                                            'deposit' => 'ডিপোজিট',
                                            'withdrawal' => 'উইথড্রয়াল',
                                            'investment' => 'বিনিয়োগ',
                                            'profit' => 'প্রফিট',
                                            'referral_bonus' => 'রেফারেল বোনাস',
                                            'admin_credit' => 'অ্যাডমিন ক্রেডিট',
                                            'admin_debit' => 'অ্যাডমিন ডেবিট'
                                        ];
                                        echo $type_labels[$transaction['type']] ?? $transaction['type'];
                                        ?>
                                    </td>
                                    <td><?php echo format_currency($transaction['amount']); ?></td>
                                    <td>
                                        <?php
                                        $status_badges = [
                                            'pending' => '<span class="badge bg-warning">পেন্ডিং</span>',
                                            'approved' => '<span class="badge bg-success">অনুমোদিত</span>',
                                            'rejected' => '<span class="badge bg-danger">প্রত্যাখ্যাত</span>'
                                        ];
                                        echo $status_badges[$transaction['status']] ?? '<span class="badge bg-secondary">অজানা</span>';
                                        ?>
                                    </td>
                                    <td><?php echo date('d/m/Y H:i', strtotime($transaction['created_at'])); ?></td>
                                    <td><?php echo htmlspecialchars(substr($transaction['description'], 0, 50)) . (strlen($transaction['description']) > 50 ? '...' : ''); ?></td>
                                </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="5" class="text-center text-muted">কোন ট্রানজেকশন পাওয়া যায়নি</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Investments -->
        <div class="card">
            <div class="card-header">
                <h5><i class="fas fa-chart-line"></i> বিনিয়োগ (সর্বশেষ ১০টি)</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>কোম্পানি</th>
                                <th>পরিমান</th>
                                <th>প্রতিদিন প্রফিট</th>
                                <th>মোট প্রফিট</th>
                                <th>অবশিষ্ট দিন</th>
                                <th>স্ট্যাটাস</th>
                                <th>তারিখ</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($investments->num_rows > 0): ?>
                                <?php while ($investment = $investments->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($investment['company_name']); ?></td>
                                    <td><?php echo format_currency($investment['amount']); ?></td>
                                    <td><?php echo format_currency($investment['profit_per_day']); ?></td>
                                    <td><?php echo format_currency($investment['total_profit']); ?></td>
                                    <td><?php echo $investment['days_remaining']; ?></td>
                                    <td>
                                        <?php
                                        $status_badges = [
                                            'active' => '<span class="badge bg-success">অ্যাক্টিভ</span>',
                                            'completed' => '<span class="badge bg-primary">সম্পন্ন</span>',
                                            'cancelled' => '<span class="badge bg-danger">বাতিল</span>'
                                        ];
                                        echo $status_badges[$investment['status']] ?? '<span class="badge bg-secondary">অজানা</span>';
                                        ?>
                                    </td>
                                    <td><?php echo date('d/m/Y', strtotime($investment['created_at'])); ?></td>
                                </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="7" class="text-center text-muted">কোন বিনিয়োগ পাওয়া যায়নি</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Balance Management Modal -->
    <div class="modal fade" id="balanceModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">ব্যালেন্স ম্যানেজমেন্ট - <span id="userName"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">বর্তমান ব্যালেন্স: <span id="currentBalance" class="fw-bold"></span></label>
                    </div>

                    <!-- Add Balance Tab -->
                    <ul class="nav nav-tabs" id="balanceTab" role="tablist">
                        <li class="nav-item" role="presentation">
                            <button class="nav-link active" id="add-tab" data-bs-toggle="tab" data-bs-target="#addBalance" type="button" role="tab">ব্যালেন্স যোগ</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="remove-tab" data-bs-toggle="tab" data-bs-target="#removeBalance" type="button" role="tab">ব্যালেন্স কাটা</button>
                        </li>
                    </ul>

                    <div class="tab-content mt-3" id="balanceTabContent">
                        <!-- Add Balance -->
                        <div class="tab-pane fade show active" id="addBalance" role="tabpanel">
                            <form id="addBalanceForm" method="POST" action="users.php">
                                <input type="hidden" name="action" value="add_balance">
                                <input type="hidden" name="user_id" id="addUserId">
                                <div class="mb-3">
                                    <label class="form-label">পরিমান (টাকা)</label>
                                    <input type="number" class="form-control" name="amount" min="0.01" step="0.01" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">কারণ</label>
                                    <textarea class="form-control" name="reason" rows="2" placeholder="ব্যালেন্স যোগের কারণ লিখুন..." required></textarea>
                                </div>
                                <button type="submit" class="btn btn-success w-100">
                                    <i class="fas fa-plus"></i> ব্যালেন্স যোগ করুন
                                </button>
                            </form>
                        </div>

                        <!-- Remove Balance -->
                        <div class="tab-pane fade" id="removeBalance" role="tabpanel">
                            <form id="removeBalanceForm" method="POST" action="users.php">
                                <input type="hidden" name="action" value="remove_balance">
                                <input type="hidden" name="user_id" id="removeUserId">
                                <div class="mb-3">
                                    <label class="form-label">পরিমান (টাকা)</label>
                                    <input type="number" class="form-control" name="amount" min="0.01" step="0.01" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">কারণ</label>
                                    <textarea class="form-control" name="reason" rows="2" placeholder="ব্যালেন্স কাটার কারণ লিখুন..." required></textarea>
                                </div>
                                <button type="submit" class="btn btn-danger w-100">
                                    <i class="fas fa-minus"></i> ব্যালেন্স কাটুন
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function showBalanceModal(userId, userName, currentBalance) {
            document.getElementById('userName').textContent = userName;
            document.getElementById('currentBalance').textContent = new Intl.NumberFormat('bn-BD', {
                style: 'currency',
                currency: 'BDT'
            }).format(currentBalance);
            document.getElementById('addUserId').value = userId;
            document.getElementById('removeUserId').value = userId;

            const modal = new bootstrap.Modal(document.getElementById('balanceModal'));
            modal.show();
        }
    </script>
</body>
</html>
