<?php
require_once '../includes/config.php';

// Simple admin check
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: index.php");
    exit();
}

// Get transaction statistics
$stmt = $conn->prepare("SELECT COUNT(*) as total FROM transactions");
$stmt->execute();
$total_transactions = $stmt->get_result()->fetch_assoc()['total'];

$stmt = $conn->prepare("SELECT COUNT(*) as pending FROM transactions WHERE status = 'pending'");
$stmt->execute();
$pending_transactions = $stmt->get_result()->fetch_assoc()['pending'];

$stmt = $conn->prepare("SELECT COUNT(*) as approved FROM transactions WHERE status = 'approved'");
$stmt->execute();
$approved_transactions = $stmt->get_result()->fetch_assoc()['approved'];

$stmt = $conn->prepare("SELECT COUNT(*) as rejected FROM transactions WHERE status = 'rejected'");
$stmt->execute();
$rejected_transactions = $stmt->get_result()->fetch_assoc()['rejected'];

$stmt = $conn->prepare("SELECT SUM(amount) as total_amount FROM transactions WHERE status = 'approved'");
$stmt->execute();
$total_amount = $stmt->get_result()->fetch_assoc()['total_amount'] ?: 0;

// Get transactions with pagination and filters
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 25;
$offset = ($page - 1) * $limit;

$status_filter = isset($_GET['status']) ? $_GET['status'] : '';
$type_filter = isset($_GET['type']) ? $_GET['type'] : '';

$query = "SELECT t.*, u.full_name as user_name FROM transactions t JOIN users u ON t.user_id = u.id";
$where_conditions = [];
$params = [];
$types = '';

if ($status_filter && in_array($status_filter, ['pending', 'approved', 'rejected'])) {
    $where_conditions[] = "t.status = ?";
    $params[] = $status_filter;
    $types .= 's';
}

if ($type_filter && in_array($type_filter, ['deposit', 'withdrawal', 'investment', 'profit', 'referral_bonus'])) {
    $where_conditions[] = "t.type = ?";
    $params[] = $type_filter;
    $types .= 's';
}

if (!empty($where_conditions)) {
    $query .= " WHERE " . implode(" AND ", $where_conditions);
}

$query .= " ORDER BY t.created_at DESC LIMIT ? OFFSET ?";
$params[] = $limit;
$params[] = $offset;
$types .= 'ii';

$stmt = $conn->prepare($query);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$transactions = $stmt->get_result();

// Get total count for pagination
$count_query = "SELECT COUNT(*) as total FROM transactions t";
if (!empty($where_conditions)) {
    $count_query .= " WHERE " . implode(" AND ", $where_conditions);
}
$count_stmt = $conn->prepare($count_query);
if (!empty($where_conditions)) {
    $count_params = array_slice($params, 0, -2); // Remove limit and offset
    $count_types = substr($types, 0, -2);
    if (!empty($count_params)) {
        $count_stmt->bind_param($count_types, ...$count_params);
    }
}
$count_stmt->execute();
$total_records = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_records / $limit);
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ট্রানজেকশন ম্যানেজমেন্ট - <?php echo $company_name; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include '../includes/admin_navbar.php'; ?>

    <div class="container mt-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2><i class="fas fa-exchange-alt"></i> ট্রানজেকশন ম্যানেজমেন্ট</h2>
            <div>
                <span class="badge bg-info fs-6">মোট: <?php echo $total_transactions; ?></span>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-warning"><?php echo $pending_transactions; ?></h4>
                        <p>পেন্ডিং</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-success"><?php echo $approved_transactions; ?></h4>
                        <p>অনুমোদিত</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-danger"><?php echo $rejected_transactions; ?></h4>
                        <p>প্রত্যাখ্যাত</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-primary"><?php echo format_currency($total_amount); ?></h4>
                        <p>মোট পরিমান</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="card mb-4">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">স্ট্যাটাস ফিল্টার</label>
                        <select name="status" class="form-control">
                            <option value="">সকল স্ট্যাটাস</option>
                            <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>পেন্ডিং</option>
                            <option value="approved" <?php echo $status_filter === 'approved' ? 'selected' : ''; ?>>অনুমোদিত</option>
                            <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>প্রত্যাখ্যাত</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">টাইপ ফিল্টার</label>
                        <select name="type" class="form-control">
                            <option value="">সকল টাইপ</option>
                            <option value="deposit" <?php echo $type_filter === 'deposit' ? 'selected' : ''; ?>>ডিপোজিট</option>
                            <option value="withdrawal" <?php echo $type_filter === 'withdrawal' ? 'selected' : ''; ?>>উইথড্রয়াল</option>
                            <option value="investment" <?php echo $type_filter === 'investment' ? 'selected' : ''; ?>>বিনিয়োগ</option>
                            <option value="profit" <?php echo $type_filter === 'profit' ? 'selected' : ''; ?>>প্রফিট</option>
                            <option value="referral_bonus" <?php echo $type_filter === 'referral_bonus' ? 'selected' : ''; ?>>রেফারেল বোনাস</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">&nbsp;</label>
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="fas fa-search"></i> ফিল্টার
                        </button>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">&nbsp;</label>
                        <a href="transactions.php" class="btn btn-secondary w-100">
                            <i class="fas fa-times"></i> ক্লিয়ার
                        </a>
                    </div>
                </form>
            </div>
        </div>

        <!-- Transactions Table -->
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>আইডি</th>
                                <th>ইউজার</th>
                                <th>টাইপ</th>
                                <th>পরিমান</th>
                                <th>স্ট্যাটাস</th>
                                <th>তারিখ</th>
                                <th>অ্যাকশন</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($transaction = $transactions->fetch_assoc()): ?>
                            <tr>
                                <td><?php echo $transaction['id']; ?></td>
                                <td><?php echo htmlspecialchars($transaction['user_name']); ?></td>
                                <td>
                                    <?php
                                    $type_labels = [
                                        'deposit' => 'ডিপোজিট',
                                        'withdrawal' => 'উইথড্রয়াল',
                                        'investment' => 'বিনিয়োগ',
                                        'profit' => 'প্রফিট',
                                        'referral_bonus' => 'রেফারেল বোনাস'
                                    ];
                                    echo $type_labels[$transaction['type']] ?? $transaction['type'];
                                    ?>
                                </td>
                                <td><?php echo format_currency($transaction['amount']); ?></td>
                                <td>
                                    <?php
                                    $status_badges = [
                                        'pending' => '<span class="badge bg-warning">পেন্ডিং</span>',
                                        'approved' => '<span class="badge bg-success">অনুমোদিত</span>',
                                        'rejected' => '<span class="badge bg-danger">প্রত্যাখ্যাত</span>'
                                    ];
                                    echo $status_badges[$transaction['status']] ?? '<span class="badge bg-secondary">অজানা</span>';
                                    ?>
                                </td>
                                <td><?php echo date('d/m/Y H:i', strtotime($transaction['created_at'])); ?></td>
                                <td>
                                    <?php if ($transaction['status'] === 'pending'): ?>
                                    <div class="btn-group" role="group">
                                        <button class="btn btn-sm btn-success" onclick="approveTransaction(<?php echo $transaction['id']; ?>)">
                                            <i class="fas fa-check"></i>
                                        </button>
                                        <button class="btn btn-sm btn-danger" onclick="rejectTransaction(<?php echo $transaction['id']; ?>)">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                    <?php else: ?>
                                    <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                <nav aria-label="Transaction pagination" class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php
                        $query_string = '';
                        if ($status_filter) $query_string .= '&status=' . $status_filter;
                        if ($type_filter) $query_string .= '&type=' . $type_filter;

                        if ($page > 1):
                        ?>
                        <li class="page-item">
                            <a class="page-link" href="?page=<?php echo $page - 1; ?><?php echo $query_string; ?>">পূর্ববর্তী</a>
                        </li>
                        <?php endif; ?>

                        <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                        <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?page=<?php echo $i; ?><?php echo $query_string; ?>"><?php echo $i; ?></a>
                        </li>
                        <?php endfor; ?>

                        <?php if ($page < $total_pages): ?>
                        <li class="page-item">
                            <a class="page-link" href="?page=<?php echo $page + 1; ?><?php echo $query_string; ?>">পরবর্তী</a>
                        </li>
                        <?php endif; ?>
                    </ul>
                </nav>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Success/Error Messages -->
    <div id="messageContainer" class="position-fixed top-0 end-0 p-3" style="z-index: 9999;"></div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/admin_transactions.js"></script>
</body>
</html>
