<?php
require_once '../includes/config.php';

// Simple admin check
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: index.php");
    exit();
}

// Get deposit statistics
$stmt = $conn->prepare("SELECT COUNT(*) as total FROM transactions WHERE type = 'deposit'");
$stmt->execute();
$total_deposits = $stmt->get_result()->fetch_assoc()['total'];

$stmt = $conn->prepare("SELECT COUNT(*) as pending FROM transactions WHERE type = 'deposit' AND status = 'pending'");
$stmt->execute();
$pending_deposits = $stmt->get_result()->fetch_assoc()['pending'];

$stmt = $conn->prepare("SELECT COUNT(*) as approved FROM transactions WHERE type = 'deposit' AND status = 'approved'");
$stmt->execute();
$approved_deposits = $stmt->get_result()->fetch_assoc()['approved'];

$stmt = $conn->prepare("SELECT COUNT(*) as rejected FROM transactions WHERE type = 'deposit' AND status = 'rejected'");
$stmt->execute();
$rejected_deposits = $stmt->get_result()->fetch_assoc()['rejected'];

$stmt = $conn->prepare("SELECT SUM(amount) as total_amount FROM transactions WHERE type = 'deposit' AND status = 'approved'");
$stmt->execute();
$total_amount = $stmt->get_result()->fetch_assoc()['total_amount'] ?: 0;

// Get deposits with pagination and filters
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 25;
$offset = ($page - 1) * $limit;

$status_filter = isset($_GET['status']) ? $_GET['status'] : '';

$query = "SELECT t.*, u.full_name as user_name, u.email as user_email FROM transactions t JOIN users u ON t.user_id = u.id WHERE t.type = 'deposit'";
$where_conditions = [];
$params = [];
$types = '';

if ($status_filter && in_array($status_filter, ['pending', 'approved', 'rejected'])) {
    $where_conditions[] = "t.status = ?";
    $params[] = $status_filter;
    $types .= 's';
}

if (!empty($where_conditions)) {
    $query .= " AND " . implode(" AND ", $where_conditions);
}

$query .= " ORDER BY t.created_at DESC LIMIT ? OFFSET ?";
$params[] = $limit;
$params[] = $offset;
$types .= 'ii';

$stmt = $conn->prepare($query);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$deposits = $stmt->get_result();

// Get total count for pagination
$count_query = "SELECT COUNT(*) as total FROM transactions t WHERE t.type = 'deposit'";
if (!empty($where_conditions)) {
    $count_query .= " AND " . implode(" AND ", $where_conditions);
}
$count_stmt = $conn->prepare($count_query);
if (!empty($where_conditions)) {
    $count_params = array_slice($params, 0, -2); // Remove limit and offset
    $count_types = substr($types, 0, -2);
    if (!empty($count_params)) {
        $count_stmt->bind_param($count_types, ...$count_params);
    }
}
$count_stmt->execute();
$total_records = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_records / $limit);
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ডিপোজিট ম্যানেজমেন্ট - <?php echo $company_name; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include '../includes/admin_navbar.php'; ?>

    <div class="container mt-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2><i class="fas fa-plus-circle"></i> ডিপোজিট ম্যানেজমেন্ট</h2>
            <div>
                <span class="badge bg-info fs-6">মোট ডিপোজিট: <?php echo $total_deposits; ?></span>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-warning"><?php echo $pending_deposits; ?></h4>
                        <p>পেন্ডিং</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-success"><?php echo $approved_deposits; ?></h4>
                        <p>অনুমোদিত</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-danger"><?php echo $rejected_deposits; ?></h4>
                        <p>প্রত্যাখ্যাত</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-center">
                    <div class="card-body">
                        <h4 class="text-primary"><?php echo format_currency($total_amount); ?></h4>
                        <p>মোট পরিমান</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="card mb-4">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">স্ট্যাটাস ফিল্টার</label>
                        <select name="status" class="form-control">
                            <option value="">সকল স্ট্যাটাস</option>
                            <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>পেন্ডিং</option>
                            <option value="approved" <?php echo $status_filter === 'approved' ? 'selected' : ''; ?>>অনুমোদিত</option>
                            <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>প্রত্যাখ্যাত</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">&nbsp;</label>
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="fas fa-search"></i> ফিল্টার
                        </button>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">&nbsp;</label>
                        <a href="deposit.php" class="btn btn-secondary w-100">
                            <i class="fas fa-times"></i> ক্লিয়ার
                        </a>
                    </div>
                </form>
            </div>
        </div>

        <!-- Deposits Table -->
        <div class="card">
            <div class="card-header">
                <h5>ডিপোজিট তালিকা</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>আইডি</th>
                                <th>ইউজার</th>
                                <th>পরিমান</th>
                                <th>পেমেন্ট মেথড</th>
                                <th>ট্রানজেকশন আইডি</th>
                                <th>স্ট্যাটাস</th>
                                <th>তারিখ</th>
                                <th>অ্যাকশন</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($deposits->num_rows > 0): ?>
                                <?php while ($deposit = $deposits->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo $deposit['id']; ?></td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($deposit['user_name']); ?></strong><br>
                                        <small class="text-muted"><?php echo htmlspecialchars($deposit['user_email']); ?></small>
                                    </td>
                                    <td><?php echo format_currency($deposit['amount']); ?></td>
                                    <td><?php echo htmlspecialchars($deposit['payment_method']); ?></td>
                                    <td><?php echo htmlspecialchars($deposit['trx_id']); ?></td>
                                    <td>
                                        <?php
                                        $status_badges = [
                                            'pending' => '<span class="badge bg-warning">পেন্ডিং</span>',
                                            'approved' => '<span class="badge bg-success">অনুমোদিত</span>',
                                            'rejected' => '<span class="badge bg-danger">প্রত্যাখ্যাত</span>'
                                        ];
                                        echo $status_badges[$deposit['status']] ?? '<span class="badge bg-secondary">অজানা</span>';
                                        ?>
                                    </td>
                                    <td><?php echo date('d/m/Y H:i', strtotime($deposit['created_at'])); ?></td>
                                    <td>
                                        <?php if ($deposit['status'] === 'pending'): ?>
                                        <div class="btn-group" role="group">
                                            <button class="btn btn-sm btn-success" onclick="approveDeposit(<?php echo $deposit['id']; ?>)">
                                                <i class="fas fa-check"></i> অনুমোদন
                                            </button>
                                            <button class="btn btn-sm btn-danger" onclick="rejectDeposit(<?php echo $deposit['id']; ?>)">
                                                <i class="fas fa-times"></i> প্রত্যাখ্যান
                                            </button>
                                        </div>
                                        <?php else: ?>
                                        <span class="text-muted">-</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="8" class="text-center text-muted">কোন ডিপোজিট পাওয়া যায়নি</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                <nav aria-label="Deposit pagination" class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php
                        $query_string = '';
                        if ($status_filter) $query_string .= '&status=' . $status_filter;

                        if ($page > 1):
                        ?>
                        <li class="page-item">
                            <a class="page-link" href="?page=<?php echo $page - 1; ?><?php echo $query_string; ?>">পূর্ববর্তী</a>
                        </li>
                        <?php endif; ?>

                        <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                        <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?page=<?php echo $i; ?><?php echo $query_string; ?>"><?php echo $i; ?></a>
                        </li>
                        <?php endfor; ?>

                        <?php if ($page < $total_pages): ?>
                        <li class="page-item">
                            <a class="page-link" href="?page=<?php echo $page + 1; ?><?php echo $query_string; ?>">পরবর্তী</a>
                        </li>
                        <?php endif; ?>
                    </ul>
                </nav>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Success/Error Messages -->
    <div id="messageContainer" class="position-fixed top-0 end-0 p-3" style="z-index: 9999;"></div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/admin.js"></script>
</body>
</html>
