<?php
header('Content-Type: text/html; charset=UTF-8');
require_once '../includes/config.php';

// Simple admin check
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: index.php");
    exit();
}

// Handle company actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        if ($_POST['action'] === 'add_company') {
            $name = trim($_POST['name']);
            $category = $_POST['category'];
            $return_percentage = (float)$_POST['return_percentage'];

            if (!empty($name) && in_array($category, ['general', 'master']) && $return_percentage > 0) {
                $stmt = $conn->prepare("INSERT INTO companies (name, category, return_percentage) VALUES (?, ?, ?)");
                $stmt->bind_param("ssd", $name, $category, $return_percentage);
                if ($stmt->execute()) {
                    $success_message = "কোম্পানি সফলভাবে যোগ হয়েছে";
                } else {
                    $error_message = "কোম্পানি যোগ করতে ব্যর্থ";
                }
            }
        } elseif ($_POST['action'] === 'update_company') {
            $id = (int)$_POST['company_id'];
            $name = trim($_POST['name']);
            $category = $_POST['category'];
            $return_percentage = (float)$_POST['return_percentage'];

            if ($id > 0 && !empty($name) && in_array($category, ['general', 'master']) && $return_percentage > 0) {
                $stmt = $conn->prepare("UPDATE companies SET name = ?, category = ?, return_percentage = ? WHERE id = ?");
                $stmt->bind_param("ssdi", $name, $category, $return_percentage, $id);
                if ($stmt->execute()) {
                    $success_message = "কোম্পানি সফলভাবে আপডেট হয়েছে";
                } else {
                    $error_message = "কোম্পানি আপডেট করতে ব্যর্থ";
                }
            }
        } elseif ($_POST['action'] === 'delete_company') {
            $id = (int)$_POST['company_id'];

            // Check if company has active investments
            $stmt = $conn->prepare("SELECT COUNT(*) as count FROM investments WHERE company_id = ? AND status = 'active'");
            $stmt->bind_param("i", $id);
            $stmt->execute();
            $active_investments = $stmt->get_result()->fetch_assoc()['count'];

            if ($active_investments > 0) {
                $error_message = "এই কোম্পানিতে অ্যাক্টিভ বিনিয়োগ আছে, ডিলিট করা যাবে না";
            } else {
                $stmt = $conn->prepare("DELETE FROM companies WHERE id = ?");
                $stmt->bind_param("i", $id);
                if ($stmt->execute()) {
                    $success_message = "কোম্পানি সফলভাবে ডিলিট হয়েছে";
                } else {
                    $error_message = "কোম্পানি ডিলিট করতে ব্যর্থ";
                }
            }
        }
    }
}

// Get all companies
$stmt = $conn->prepare("SELECT * FROM companies ORDER BY category, name");
$stmt->execute();
$companies = $stmt->get_result();
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>কোম্পানি ম্যানেজমেন্ট - <?php echo $company_name; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include '../includes/admin_navbar.php'; ?>

    <div class="container mt-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2><i class="fas fa-building"></i> কোম্পানি ম্যানেজমেন্ট</h2>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCompanyModal">
                <i class="fas fa-plus"></i> নতুন কোম্পানি যোগ করুন
            </button>
        </div>

        <?php if (isset($success_message)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?php echo $success_message; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?php echo $error_message; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <!-- Companies Table -->
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>আইডি</th>
                                <th>কোম্পানি নাম</th>
                                <th>ক্যাটাগরি</th>
                                <th>রিটার্ন (%)</th>
                                <th>তৈরির তারিখ</th>
                                <th>অ্যাকশন</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($company = $companies->fetch_assoc()): ?>
                            <tr>
                                <td><?php echo $company['id']; ?></td>
                                <td><?php echo htmlspecialchars($company['name']); ?></td>
                                <td>
                                    <span class="badge bg-<?php echo $company['category'] === 'general' ? 'primary' : 'success'; ?>">
                                        <?php echo $company['category'] === 'general' ? 'সাধারণ' : 'মাস্টার'; ?>
                                    </span>
                                </td>
                                <td><?php echo $company['return_percentage']; ?>%</td>
                                <td><?php echo date('d/m/Y', strtotime($company['created_at'])); ?></td>
                                <td>
                                    <div class="btn-group" role="group">
                                        <button class="btn btn-sm btn-warning" onclick="editCompany(<?php echo $company['id']; ?>, '<?php echo addslashes($company['name']); ?>', '<?php echo $company['category']; ?>', <?php echo $company['return_percentage']; ?>)">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button class="btn btn-sm btn-danger" onclick="deleteCompany(<?php echo $company['id']; ?>, '<?php echo addslashes($company['name']); ?>')">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Company Modal -->
    <div class="modal fade" id="addCompanyModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">নতুন কোম্পানি যোগ করুন</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="addCompanyForm" method="POST" accept-charset="UTF-8">
                        <input type="hidden" name="action" value="add_company">
                        <div class="mb-3">
                            <label class="form-label">কোম্পানি নাম</label>
                            <input type="text" class="form-control" name="name" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">ক্যাটাগরি</label>
                            <select class="form-control" name="category" required>
                                <option value="general">সাধারণ বিনিয়োগ</option>
                                <option value="master">মাস্টার বিনিয়োগ</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">রিটার্ন পার্সেন্টেজ (%)</label>
                            <input type="number" class="form-control" name="return_percentage" step="0.01" min="0.01" max="100" required>
                        </div>
                        <button type="submit" class="btn btn-primary w-100">যোগ করুন</button>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Company Modal -->
    <div class="modal fade" id="editCompanyModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">কোম্পানি আপডেট করুন</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="editCompanyForm" method="POST" accept-charset="UTF-8">
                        <input type="hidden" name="action" value="update_company">
                        <input type="hidden" name="company_id" id="editCompanyId">
                        <div class="mb-3">
                            <label class="form-label">কোম্পানি নাম</label>
                            <input type="text" class="form-control" name="name" id="editCompanyName" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">ক্যাটাগরি</label>
                            <select class="form-control" name="category" id="editCompanyCategory" required>
                                <option value="general">সাধারণ বিনিয়োগ</option>
                                <option value="master">মাস্টার বিনিয়োগ</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">রিটার্ন পার্সেন্টেজ (%)</label>
                            <input type="number" class="form-control" name="return_percentage" id="editCompanyReturn" step="0.01" min="0.01" max="100" required>
                        </div>
                        <button type="submit" class="btn btn-warning w-100">আপডেট করুন</button>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Company Modal -->
    <div class="modal fade" id="deleteCompanyModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">কোম্পানি ডিলিট করুন</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p>আপনি কি নিশ্চিত যে <strong id="deleteCompanyName"></strong> কোম্পানিটি ডিলিট করতে চান?</p>
                    <div class="alert alert-warning">
                        <strong>সতর্কতা:</strong> এই কোম্পানিতে অ্যাক্টিভ বিনিয়োগ থাকলে ডিলিট করা যাবে না।
                    </div>
                    <form id="deleteCompanyForm" method="POST" accept-charset="UTF-8">
                        <input type="hidden" name="action" value="delete_company">
                        <input type="hidden" name="company_id" id="deleteCompanyId">
                        <div class="d-flex justify-content-end">
                            <button type="button" class="btn btn-secondary me-2" data-bs-dismiss="modal">বাতিল</button>
                            <button type="submit" class="btn btn-danger">ডিলিট করুন</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/admin_companies.js"></script>
</body>
</html>
